/**
 * Copyright (c) 2009 - Lehigh University. Bethlehem, PA, USA.
 * All rights reserved.
 * This source code is a part of MARCHES Project. 
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL LEHIGH UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT, 
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF LEHIGH
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *	
 * LEHIGH UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, 
 * AND LEHIGH UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 **/

// modsDlg.cpp : implementation file
//

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Collections;
using System.Reflection;

namespace MassWare
{
    internal class XMLParser
    {
        #region variables
        public const string strTypeHeader = "System.";

        private string m_strXmlFile;
        private Hashtable m_tbMasstools, m_tbProMasslets, m_tbReMasslets;
        private Architecture m_arch;

        private Hashtable m_tbEvents, m_tbVMasslets; // local tables for events and virtual components
        private XmlTextReader m_xReader;

        #endregion

        #region constructors and public variables
        public XMLParser()
        {
            m_strXmlFile = "dc.xml";
            m_tbEvents = new Hashtable();
            m_tbVMasslets = new Hashtable();
        }
        public XMLParser(string strfile)
        {
            m_strXmlFile = strfile;
            m_tbEvents = new Hashtable();
            m_tbVMasslets = new Hashtable();
        }

        public void Dispose()
        {
            m_tbEvents.Clear();
            m_tbVMasslets.Clear();
        }

        public Architecture mArchitecture
        {
            get
            {
                return m_arch;
            }
            set
            {
                m_arch = value;
            }
        }
        public void SetArch(Architecture arch)
        {
            m_arch = arch;
            m_tbMasstools = arch.mMassTools;
            m_tbProMasslets = arch.mProMasslets;
            m_tbReMasslets = arch.mReMasslets;
        }
        public void SetXmlFile(string strfile)
        {
            m_strXmlFile = strfile;
        }
        #endregion

        #region self-defined functions
        public void Start()
        {
            m_xReader = new XmlTextReader(m_strXmlFile);
            m_xReader.WhitespaceHandling = WhitespaceHandling.None;
            Parse();
            m_xReader.Close();
        }

        public bool Parse()
        {
            if (m_xReader.MoveToContent() == XmlNodeType.Element)
                m_xReader.ReadStartElement("DecisionEngine");
            if (!ParseInformationServer()) return false;
            if (!ParseMassTools()) return false;
            if (!ParseMasslets()) return false;
            if (!ParseRules()) return false;
            m_xReader.ReadEndElement(); // end DecisionEngine
            return true;
        }

        public bool ParseInformationServer()
        {
            string ipAddr;
            int port;
            if (m_xReader.MoveToContent() == XmlNodeType.Element)
                m_xReader.ReadStartElement("InformationServer");
            m_xReader.ReadStartElement("ip");
            ipAddr = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement(); // end ip
            m_xReader.ReadStartElement("port");
            port = m_xReader.ReadContentAsInt();
            m_xReader.ReadEndElement(); // end port
            m_xReader.ReadEndElement(); // end InformationServer
            return true;
        }

        public bool ParseMassTools()
        {
            if (m_xReader.MoveToContent() != XmlNodeType.Element) return false;
            if (m_xReader.Name != "MassTools") return true;

            if (m_xReader.ReadToFollowing("component"))
            {
                do
                {
                    if (!ParseSingleMassTool()) return false;
                } while (m_xReader.ReadToNextSibling("component"));
            }
            m_xReader.Skip();
            return true;
        }

        public bool ParseSingleMassTool()
        {
            m_xReader.ReadStartElement("component");

            string file, name, type, alias, value;
            MassWare.MassToolReflectObject rfob = new MassWare.MassToolReflectObject();
            Assembly a = null; 

            if (m_xReader.Name == "file")
            {
                m_xReader.ReadStartElement("file");
                file = m_xReader.ReadString().Trim();
                if (file.IndexOf('/') > 0 || file.IndexOf('\\') > 0)
                    a = Assembly.LoadFile(file);
                else
                    a = Assembly.LoadFrom(file);

                m_xReader.ReadEndElement();
            }

            m_xReader.ReadStartElement("name");
            name = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement();

            // parse type and alias
            m_xReader.ReadStartElement("vtype");
            type = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement();

            rfob.name = name;
            //////////////////////////add subscriber function
            if (m_xReader.Name == "SubFun")
            {
                m_xReader.ReadStartElement("SubFun");
                rfob.outfunc = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();
            }
            else rfob.outfunc = "AddSubscriber";
            ////////////////////default is AddSubscriber

            m_xReader.ReadStartElement("alias");
            alias = m_xReader.ReadString().Trim();

            if (a == null) // shared assembly, registered
            {
                rfob.type = Type.GetTypeFromProgID(name);
                rfob.obj = Activator.CreateInstance(rfob.type);
            }
            else  // local assembly, no registered
            {
                rfob.type = a.GetType(name);
                rfob.obj = a.CreateInstance(name);
            }

            // parse parameters
            while (m_xReader.ReadToNextSibling("param"))
            {
                m_xReader.ReadToDescendant("name");

                m_xReader.ReadStartElement("name");
                name = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();

                m_xReader.ReadStartElement("vtype");
                type = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();

                m_xReader.ReadStartElement("value");
                value = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();

                // set parameters
                Type t = Type.GetType(strTypeHeader + type);
                Object[] p = { Convert.ChangeType(value, t) };
                
                rfob.type.InvokeMember(name, BindingFlags.InvokeMethod, null, rfob.obj, p);
            }
            
            m_tbMasstools.Add(alias, rfob);
            return true;
        }

        public bool ParseMasslets()
        {
            if (m_xReader.MoveToContent() != XmlNodeType.Element) return false;
            if (m_xReader.Name != "Masslets") return true;

            if (m_xReader.ReadToFollowing("component"))
            {
                do
                {
                    if (!ParseSingleMasslet()) return false;
                } while (m_xReader.ReadToNextSibling("component"));
            }
            m_xReader.Skip();
            return true;
        }

        public bool ParseSingleMasslet()
        {
            m_xReader.ReadStartElement("component");

            string file, name, type, alias;
            MassWare.MassletReflectObject rfob = new MassWare.MassletReflectObject();
            Assembly a = null;

            if (m_xReader.Name == "file") // local assembly, no registered
            {
                m_xReader.ReadStartElement("file");
                file = m_xReader.ReadString().Trim();
                if (file.IndexOf('/') > 0 || file.IndexOf('\\') > 0)
                    a = Assembly.LoadFile(file);
                else
                    a = Assembly.LoadFrom(file);

                m_xReader.ReadEndElement();
            }

            m_xReader.ReadStartElement("name");
            name = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement();

            m_xReader.ReadStartElement("atype");
            type = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement();

            m_xReader.ReadStartElement("alias");
            alias = m_xReader.ReadString().Trim();

            rfob.name = name;
            // dont creat instances for virtual components
            if ("proActive" == type || "reActive" == type) 
            {
                if (a == null) // shared assembly, registered
                {
                    rfob.type = Type.GetTypeFromProgID(name);
                    rfob.obj = Activator.CreateInstance(rfob.type);
                }
                else
                {
                    rfob.type = a.GetType(name);
                    rfob.obj = a.CreateInstance(name);
                }

            }

            // parse parameters
            rfob.tbparam = new Hashtable();
            MassWare.ParamObject cparam;
            while (m_xReader.ReadToNextSibling("param"))
            {
                m_xReader.ReadToDescendant("name");
                m_xReader.ReadStartElement("name");
                name = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();

                m_xReader.ReadStartElement("vtype");
                cparam.type = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement();

                cparam.value = "";
                if (m_xReader.Name == "value")
                {
                    m_xReader.ReadStartElement("value");
                    cparam.value = m_xReader.ReadString().Trim();
                    m_xReader.ReadEndElement();

                    Type t = Type.GetType(strTypeHeader + cparam.type);
                    Object[] p = { Convert.ChangeType(cparam.value, t) };
                    rfob.type.InvokeMember(name, BindingFlags.InvokeMethod, null, rfob.obj, p);
                }

                rfob.tbparam.Add(name, cparam);
            }
            if ("proActive" == type)
                m_tbProMasslets.Add(alias, rfob);
            else if ("reActive" == type)
                m_tbReMasslets.Add(rfob.name, rfob); // for reactive component, we use name instead of alias
            else if ("virtual" == type)
                m_tbVMasslets.Add(alias, rfob);
            //else if ("both" == type.Trim()) ; // does not accept both now

            return true;
        }

        public bool ParseRules()
        {
            if (m_xReader.MoveToContent() != XmlNodeType.Element) return false;
            if (m_xReader.Name != "Rules") return true;

            if (m_xReader.ReadToFollowing("rule"))
            {
                do
                {
                    if (!ParseSingleRule()) return false;
                } while (m_xReader.ReadToNextSibling("rule"));
            }
            m_xReader.Skip();
            return true;
        }

        public bool ParseSingleRule()
        {
            EventBC cond;
            Actuator pact, ract = null;
            m_xReader.ReadStartElement("rule");
            cond = ParseCondition();
            pact = ParseActuator();
            if (m_xReader.Name == "Actuator")
                ract = ParseActuator();
            m_arch.AddAcutator(cond, pact, ract);
            return true;
        }

        public EventBC ParseCondition()
        {
            m_xReader.ReadStartElement("condition");
            EventBC res = ParseOperation();
            m_xReader.ReadEndElement();
            return res;
        }

        public EventBC ParseOperation()
        {
            m_xReader.ReadStartElement("operation");

            string strtype, strexpr, strkey;
            EventBC bc = new EventBC(), bclhs = new EventBC(), bcrhs = new EventBC();
            EventExpr exprlhs = new EventExpr(), exprrhs = new EventExpr();
            
            m_xReader.ReadStartElement("otype");
            strtype = m_xReader.ReadString().Trim().ToUpper();
            m_xReader.ReadEndElement(); // end otype

            m_xReader.ReadStartElement("lhs");
            m_xReader.MoveToContent(); // get the name, but not read it
            if ("operation" == m_xReader.Name)
            {
                bclhs = ParseOperation();
            }
            else if ("expr" == m_xReader.Name)
            {
                m_xReader.ReadStartElement("expr");
                strexpr = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement(); //end expr
                exprlhs = EventExpr.BuildEventExpr(strexpr, m_tbMasstools, m_tbEvents);

            }
            m_xReader.ReadEndElement(); // end lhs

            m_xReader.ReadStartElement("rhs");
            m_xReader.MoveToContent(); // get the name but not read it
            if ("operation" == m_xReader.Name)
            {
                bcrhs = ParseOperation();
            }
            else if ("expr" == m_xReader.Name)
            {
                m_xReader.ReadStartElement("expr");
                strexpr = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement(); //end expr
                exprrhs = EventExpr.BuildEventExpr(strexpr, m_tbMasstools, m_tbEvents);
            }
            m_xReader.ReadEndElement(); // end rhs
            m_xReader.ReadEndElement(); // end operation


            if ("AND" == strtype || "OR" == strtype)
            {
                strkey = EventBool.CreateKey(bclhs, bcrhs, strtype);
                if (m_tbEvents.ContainsKey(strkey)) return (EventBool)m_tbEvents[strkey];

                bc = EventBool.MergeEvents(bclhs, bcrhs, strtype);
                m_tbEvents.Add(strkey, bc);
            }
            else
            {
                strkey = EventComp.CreateKey(exprlhs, exprrhs, strtype);
                if (m_tbEvents.ContainsKey(strkey)) return (EventComp)m_tbEvents[strkey];

                bc = EventComp.MergeEvents(exprlhs, exprrhs, strtype);
                m_tbEvents.Add(strkey, bc);
            }

            return bc;
        }

        public Actuator ParseActuator()
        {
            m_xReader.ReadStartElement("Actuator");

            Actuator act = new Actuator();
            string atype, method, param, arch;
            int ntype; // 0 for proactive, 1 for reactive

            m_xReader.ReadStartElement("atype");
            atype = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement(); //end atype
            if ("proActive" == atype) ntype = 0;
            else if ("reActive" == atype) ntype = 1;
            else throw new System.Exception("no such architecture type!");

            m_xReader.ReadStartElement("method");
            method = m_xReader.ReadString().Trim();
            m_xReader.ReadEndElement(); //end method
            act.SetSyncType(method);

            param = "";
            if (m_xReader.Name == "SetParam")
            {
                m_xReader.ReadStartElement("SetParam");
                param = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement(); //end param                
            }

            if (m_xReader.Name == "SetArch")
            {
                m_xReader.ReadStartElement("SetArch");
                arch = m_xReader.ReadString().Trim();
                m_xReader.ReadEndElement(); //end arch

                if (ntype == 0) ParseArchString(arch, act, m_tbProMasslets);
                else if (ntype == 1) ParseArchString(arch, act, m_tbVMasslets);
            }

            if (param.Trim('\r', '\n', ' ', '\t').Length > 0)
            {
                if (ntype == 0) ParseArchParam(param, act, m_tbProMasslets);
                else if (ntype == 1) ParseArchParam(param, act, m_tbVMasslets);
            }

            m_xReader.ReadEndElement(); //end actuator
            return act;
        }


        #endregion

        #region parse arch and parameter strings
        private static void ParseArchString(string arch, Actuator act, Hashtable tb)
        {
            List<MassWare.MassletReflectObject> list = act.mActuator;
            string[] conn = arch.Split(';');
            char[] charsplitter = { '.' };
            string[] strsplitter = { "->" };
            List<string> startname = new List<string>(), startfun = new List<string>();

            foreach (string str in conn)
            {
                string orgstr = str.Trim('\r', '\n', ' ', '\t');
                if (orgstr.Length > 0)
                {
                    string[] Comps = orgstr.Split(strsplitter, StringSplitOptions.None);
                    if (Comps.Length == 1)
                    {
                        string[] startComp = Comps[0].Split(charsplitter);
                        startname.Add(((MassWare.MassletReflectObject)tb[startComp[0].Trim()]).name);
                        startfun.Add(startComp[1].Trim());
                        continue;
                    }
                    string[] comp1 = Comps[0].Split(charsplitter);
                    string[] comp2 = Comps[1].Split(charsplitter);

                    int idx1 = -1, idx2 = -1;
                    MassWare.MassletReflectObject obj1 = new MassWare.MassletReflectObject(), 
                        obj2 = new MassWare.MassletReflectObject();
                    MassWare.Connector connIn = new MassWare.Connector(), 
                        connOut = new MassWare.Connector();

                    if (comp1.Length > 1)
                    {
                        if (!tb.ContainsKey(comp1[0].Trim()))
                            throw new System.Exception("archtecture contains an undefined masslet");

                        obj1 = (MassWare.MassletReflectObject)tb[comp1[0].Trim()];

                        for (int i = 0; i < list.Count; i++)
                            // if the list already contains comp1
                            if (list[i].name == obj1.name)
                            {
                                idx1 = i;
                                break;
                            }
                        if (idx1 != -1) obj1 = list[idx1];
                        else obj1 = InitializeObj(obj1);

                        connOut.interf = GetNameSpace(obj1.name) + comp1[1].Trim();
                        if (comp1.Length > 2) connOut.func = comp1[2].Trim();
                        else connOut.func = "AddSubscriber";
                    }
                    else obj1.name = comp1[0].Trim();

                    if (comp2.Length > 1)
                    {
                        if (!tb.ContainsKey(comp2[0].Trim()))
                            throw new System.Exception("archtecture contains an undefined masslet");

                        obj2 = (MassWare.MassletReflectObject)tb[comp2[0].Trim()];
                        for (int i = 0; i < list.Count; i++)
                            // if the list already contains comp1
                            if (list[i].name == obj2.name)
                            {
                                idx2 = i;
                                break;
                            }
                        if (idx2 != -1) obj2 = list[idx2];
                        else obj2 = InitializeObj(obj2);

                        connIn.interf = GetNameSpace(obj2.name) + comp2[1].Trim();
                        if (comp2.Length > 2) connIn.func = comp2[2].Trim();
                        else connIn.func = "GetSubscriber";
                    }
                    else obj2.name = comp2[0].Trim();

                    if (comp1.Length > 1)
                    {
                        obj1.tbOutput.Add(obj2.name, connOut);
                        if (idx1 != -1) list[idx1] = obj1;
                        else list.Add(obj1);
                    }

                    if (comp2.Length > 1)
                    {
                        obj2.tbInput.Add(obj1.name, connIn);
                        if (idx2 != -1) list[idx2] = obj2;
                        else list.Add(obj2);
                    }
                }
            }
            foreach (MassWare.MassletReflectObject rfob in list)
            {
                if (startname.Contains(rfob.name))
                {
                    MassWare.Connector con = new MassWare.Connector();
                    con.func = startfun[startname.IndexOf(rfob.name)];
                    con.interf = "Stop";
                    rfob.tbInput.Add(Actuator.START_INTERFACE, con);
                }
            }
        }

        private static void ParseArchParam(string param, Actuator act, Hashtable tb)
        {
            List<MassWare.MassletReflectObject> list = act.mActuator;
            string[] paramstrs = param.Split(';');
            foreach (string str in paramstrs)
            {
                string orgstr = str.Trim('\r', '\n', ' ', '\t');
                if (orgstr.Length > 0)
                {
                    string[] twoparts = orgstr.Split('=');
                    string[] part1 = twoparts[0].Split('.');

                    MassWare.MassletReflectObject targetOb = (MassWare.MassletReflectObject)tb[part1[0].Trim()];

                    string name = targetOb.name;
                    string func = part1[1].Trim();

                    MassWare.ParamObject pobj = (MassWare.ParamObject)targetOb.tbparam[func];
                    pobj.value = twoparts[1].Trim();

                    for (int i = 0; i < list.Count; i++)
                        if (list[i].name == name)
                        {
                            list[i].tbparam.Add(func, pobj);
                            break;
                        }
                }
            }
        }

        private static MassWare.MassletReflectObject InitializeObj(MassWare.MassletReflectObject rfob)
        {
            rfob.tbInput = new Hashtable();
            rfob.tbOutput = new Hashtable();
            rfob.tbparam = new Hashtable();
            return rfob;
        }

        private static string GetNameSpace(string name)
        {
            return name.Remove(name.LastIndexOf('.') + 1);
        }

        #endregion
    }
}
