/**
 * Copyright (c) 2009 - Lehigh University. Bethlehem, PA, USA.
 * All rights reserved.
 * This source code is a part of MARCHES Project. 
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL LEHIGH UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT, 
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF LEHIGH
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *	
 * LEHIGH UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, 
 * AND LEHIGH UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 **/

// modsDlg.cpp : implementation file
//

using System;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.Sockets;

namespace MassWare
{
    public class UDPServer //internal, public for X.doc generation
    {
        #region self defined structure
        public class StateObjectUdp
        {
            public Socket sock = null;
            public EndPoint remoteEndPoint;
            public const int BUFFER_SIZE = 65536;
            public byte[] dataBuffer = new byte[BUFFER_SIZE];
        }
        #endregion

        public const int PORTVIDEO = 5365, PORTAUDIO = 5366, PORTSLIDE = 5368;
        private Socket m_sock;
        private int m_nPort;

        public AsyncCallback pfnReceivingCallBack;

        public delegate void UDPDataReadyEventHandler(object sender, UDPEventArgs e); 
        // Delegate instance for data ready and output data from DataOutput interface
        public event UDPDataReadyEventHandler DataReadyEvent;

        public UDPServer()
        {
            m_nPort = 7788;
        }

        public UDPServer(int port)
        {
            m_nPort = port;
        }

        public Socket m_socket
        {
            get
            {
                return m_sock;
            }
        }

        public void WaitForUDPData(Socket sc)
        {
            try
            {
                IPEndPoint remoteIpPoint = new IPEndPoint(IPAddress.Any, 0);
                EndPoint clientEndPoint = (EndPoint)remoteIpPoint;

                StateObjectUdp udpPkt = new StateObjectUdp();
                udpPkt.sock = sc;
                udpPkt.remoteEndPoint = clientEndPoint;
                if (pfnReceivingCallBack == null)
                {
                    pfnReceivingCallBack = new AsyncCallback(OnUDPDataReceived);
                }
                m_sock.BeginReceiveFrom(udpPkt.dataBuffer, 0, StateObjectUdp.BUFFER_SIZE, SocketFlags.None, ref clientEndPoint,
                    pfnReceivingCallBack, udpPkt);
            }
            catch (SocketException se)
            {
                Console.WriteLine(se.Message);
            }
        }

        public void OnUDPDataReceived(IAsyncResult asyn)
        {
            try
            {
                StateObjectUdp udpPkt = (StateObjectUdp)asyn.AsyncState;

                // Complete the BeginReceive() asynchronous call by EndReceive() method
                // which will return the number of characters written to the stream 
                // by the client
                int iRx = udpPkt.sock.EndReceiveFrom(asyn, ref udpPkt.remoteEndPoint);

                if (DataReadyEvent != null)
                    ProcessData(udpPkt, iRx);

                // Continue the waiting for data on the Socket
                WaitForUDPData(udpPkt.sock);
            }
            catch (ObjectDisposedException)
            {
                System.Diagnostics.Debugger.Log(0, "1", "\nOnDataReceived: Socket has been closed\n");
            }
            catch (SocketException se)
            {
                Console.WriteLine(se.Message);
            }
        }

        private void ProcessData(StateObjectUdp spk, int revlen)
        {
            byte type = spk.dataBuffer[0];

            byte[] databyte = null;
            if (revlen > 1)
            {
                databyte = new byte[revlen - 1];
                for (int i = 0; i < revlen - 1; i++)
                    databyte[i] = spk.dataBuffer[i + 1];
            }

            DataReadyEvent(this, new UDPEventArgs(type, databyte, ((IPEndPoint)spk.remoteEndPoint).Address));
        }

        #region override functions
        public bool Start()
        {
            m_sock = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
            IPEndPoint ipPoint = new IPEndPoint(IPAddress.Any, m_nPort);
            m_sock.Bind(ipPoint);
            WaitForUDPData(m_sock);
            return true;
        }

        public bool Close()
        {
            if (m_sock != null)
            {
                // m_sockChbox.Disconnect(true);
                m_sock.Close();
                m_sock = null;
            }
            return true;
        }
        #endregion

        #region set parameters
        public void SetPortNumber(int port)
        {
            m_nPort = port;
        }
        #endregion


    }
    
    public class UDPEventArgs : EventArgs
    {
        private byte type;
        private byte[] data;
        private IPAddress ipaddr;
        public UDPEventArgs(byte tp, byte[] dt, IPAddress addr)
        {
            type = tp;
            data = dt;
            ipaddr = addr;
        }
        public byte m_type
        {
            get
            {
                return type;
            }
        }
        public byte[] m_data
        {
            get
            {
                return data;
            }
        }
        public IPAddress m_ipAddr
        {
            get
            {
                return ipaddr;
            }
        }
    }
}
