/**
 * Copyright (c) 2009 - Lehigh University. Bethlehem, PA, USA.
 * All rights reserved.
 * This source code is a part of MARCHES Project. 
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL LEHIGH UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT, 
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF LEHIGH
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *	
 * LEHIGH UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, 
 * AND LEHIGH UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 **/

// modsDlg.cpp : implementation file
//

using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;

namespace MassWare
{
    public class ExprParser //internal, public for X.doc generation
    {
        enum TOKENTYPE { VARIABLE = 0, NUMBER, ADD, SUB, MUL, DIV, LPAR, RPAR, JUNK };
        Hashtable tbToken = new Hashtable();
        Hashtable tbTools;
        Hashtable tbEvents;

        string m_expr, m_tok;
        char m_ch;
        TOKENTYPE m_type;
        int n_pos;
        double m_value;

        public ExprParser(string str, Hashtable tb)
        {
            m_expr = str;
            n_pos = 0;
            tbTools = tb;
            tbToken.Add("+", TOKENTYPE.ADD);
            tbToken.Add("-", TOKENTYPE.SUB);
            tbToken.Add("*", TOKENTYPE.MUL);
            tbToken.Add("/", TOKENTYPE.DIV);
            tbToken.Add("(", TOKENTYPE.LPAR);
            tbToken.Add(")", TOKENTYPE.RPAR);
        }

        public void SetEventTable(Hashtable tb)
        {
            tbEvents = tb;
        }

        public EventExpr Parse()
        {
            m_type = Next();
            return S();         
        }

        private TOKENTYPE Next()
        {
            if (n_pos == 0) m_ch = m_expr[n_pos++];
            stripWhite();

            m_tok = "";
            char cc = m_ch; // cc is used to judge what type of the token

            // if the current char is a operator such as '()+-\n..'
            // it is a token and we don't need to get the next charactor
            // else if current char is 'A'~'Z' or '0'~'9', get next and check
            if ((m_ch >= 'A' && m_ch <= 'Z') || (m_ch >= 'a' && m_ch <= 'z'))
                do
                {
                    m_tok += m_ch;
                    if (n_pos < m_expr.Length)
                        m_ch = m_expr[n_pos++];
                    else
                    {
                        m_ch = '\0';
                        break;
                    }
                } while ((m_ch >= 'A' && m_ch <= 'Z') || (m_ch >= 'a' && m_ch <= 'z') || (m_ch >= '0' && m_ch <= '9'));
            else if ((m_ch >= '0' && m_ch <= '9') || (m_ch == '.'))
                do
                {
                    m_tok += m_ch;
                    if (n_pos < m_expr.Length)
                        m_ch = m_expr[n_pos++];
                    else
                    {
                        m_ch = '\0';
                        break;
                    }
                } while ((m_ch >= '0' && m_ch <= '9') || (m_ch == '.'));
            else
            {
                m_tok += m_ch;
                if (n_pos < m_expr.Length)
                    m_ch = m_expr[n_pos++];
                else m_ch = '\0';  
            }

            // to judge the type of the token 
            if ((cc >= 'A' && cc <= 'Z') || (cc >= 'a' && cc <= 'z'))
            {
                if (tbToken.ContainsKey(m_tok)) return (TOKENTYPE)tbToken[m_tok];
                else return TOKENTYPE.VARIABLE;
            }
            else if ((cc >= '0' && cc <= '9') || cc == '.')
            {
                m_value = Double.Parse(m_tok);
                return TOKENTYPE.NUMBER;
            }
            else
            {
                if (tbToken.ContainsKey(m_tok)) return (TOKENTYPE)tbToken[m_tok];
            }
            return TOKENTYPE.JUNK;

        }

        private void stripWhite()
        {
            while (m_ch == ' ' || m_ch == '\t' || m_ch == '\r' || m_ch == '\n')
                m_ch = m_expr[n_pos++];
        }


        private EventExpr S()
        {
            EventExpr temp;
            EventExpr result = T();

            while (m_type == TOKENTYPE.ADD || m_type == TOKENTYPE.SUB)
            {
                char type = m_tok[0];
                m_type = Next();
                temp = T();
                string key = EventExpr.CreateKey(result, temp, type);
                if (tbEvents.ContainsKey(key)) //DAG
                    result = (EventExpr)tbEvents[key];
                else
                {
                    result = EventExpr.MergeEvents(result, temp, type);
                    tbEvents.Add(key, result);
                }
            }
            return result;
        }

        private EventExpr T()
        {
            EventExpr temp;
            EventExpr result = F();

            while (m_type == TOKENTYPE.MUL || m_type == TOKENTYPE.DIV)
            {
                char type = m_tok[0];
                m_type = Next();
                temp = F();
                string key = EventExpr.CreateKey(result, temp, type);
                if (tbEvents.ContainsKey(key))
                    result = (EventExpr)tbEvents[key]; //DAG
                else
                {
                    result = EventExpr.MergeEvents(result, temp, type);
                    tbEvents.Add(key, result);
                }
            }
            return result;
        }

        private EventExpr F()
        {
            EventExpr result;
            switch (m_type)
            {
                case TOKENTYPE.LPAR:
                    m_type = Next();
                    result = S();
                    if (m_type != TOKENTYPE.RPAR) 
                        throw new System.Exception("expr error: ')' expected.");
                    break;
                case TOKENTYPE.NUMBER:
                    result = new EventExpr(m_value);
                    break;
                case TOKENTYPE.VARIABLE:
                    if (!tbEvents.ContainsKey(m_tok))
                    {
                        if (!tbTools.ContainsKey(m_tok))
                            throw new System.Exception("the measurement tool: " + m_tok + " is not defined");
                        result = new EventExpr((MassWare.MassToolReflectObject)tbTools[m_tok], m_tok);
                        tbEvents.Add(result.mPositiveTag, result);
                    }
                    else result = (EventExpr)tbEvents[m_tok];
                    break;
                default: 
                    throw new System.Exception("expr error: NUM, VARIABLE, '(' expected.");
            }
            m_type = Next();
            return result;
        }
        
    }
}
