/**
 * Copyright (c) 2009 - Lehigh University. Bethlehem, PA, USA.
 * All rights reserved.
 * This source code is a part of MARCHES Project. 
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL LEHIGH UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT, 
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF LEHIGH
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *	
 * LEHIGH UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, 
 * AND LEHIGH UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 **/

// modsDlg.cpp : implementation file
//

using System;
using System.Collections.Generic;
using System.Text;
using MassTools;
using System.Collections;
using System.Reflection;

namespace MassWare
{
    // could be a single masstool or an expression of a eventleaf
    public class EventExpr //internal, public for X.doc generation
    {
        #region variables and constructors
        public enum OPTYPE {OP_NONE = 0, OP_ADD, OP_SUB, OP_MUL, OP_DIV};
        public enum SIDETYPE { SD_SOLO = 0, SD_LEFT, SD_RIGHT };

        // Delegate instance for data ready and output data to subscribers
        public event MassToolDataReadyEventHandler EventDataReadyEvent;

        // Delegate instance for the masstool callback - must keep alive! gc should NOT collect it
        private MassToolDataReadyEventHandler mToolEventHandler, 
            mLeftEventHandler, // for left hand expression
            mRightEventHandler; // for right hand expression

        private double m_dValue, // the output of this leaf
            m_dLeft,             // the left hand value
            m_dRight;            // the right hand value
        private OPTYPE opType;   // the operation type.

        private string m_posTag, m_negTag;
                
        public EventExpr()
        {
            opType = OPTYPE.OP_NONE;
            EventDataReadyEvent = null;
            mToolEventHandler = null;
            mLeftEventHandler = null;
            mRightEventHandler = null;
        }

        public EventExpr(double value)
        {
            opType = OPTYPE.OP_NONE;
            m_posTag = "" + value;
            m_dValue = value;

            EventDataReadyEvent = null;
            mToolEventHandler = null;
            mLeftEventHandler = null;
            mRightEventHandler = null;
        }

        public EventExpr(MassWare.MassToolReflectObject rfob, string tag)
        {            
            opType = OPTYPE.OP_NONE;
            m_posTag = tag;
            object[] regArg = { GetListener(SIDETYPE.SD_SOLO) };
            rfob.type.InvokeMember(
                rfob.outfunc,
                BindingFlags.InvokeMethod,  // how to bind
                null,                       // binder
                rfob.obj,                    // the COM object
                regArg);           // the method arguments

            EventDataReadyEvent = null;
            mLeftEventHandler = null;
            mRightEventHandler = null;
        }

        #endregion

        #region event and call back functions
        public void AddSubscriber(MassToolDataReadyEventHandler handler)
        {
            EventDataReadyEvent += handler;
        }

        public MassToolDataReadyEventHandler GetListener(SIDETYPE t)
        {
            switch (t)
            {
                case SIDETYPE.SD_SOLO:
                    if (mToolEventHandler == null)
                        mToolEventHandler = new MassToolDataReadyEventHandler(ToolDataReceiveCallBack);
                    return mToolEventHandler;
                case SIDETYPE.SD_LEFT:
                    if (mLeftEventHandler == null)
                        mLeftEventHandler = new MassToolDataReadyEventHandler(LeftDataReceiveCallBack);
                    return mLeftEventHandler;
                case SIDETYPE.SD_RIGHT:
                    if (mRightEventHandler == null)
                        mRightEventHandler = new MassToolDataReadyEventHandler(RightDataReceiveCallBack);
                    return mRightEventHandler;
                default:
                    return null;
            }
        }

        public void ToolDataReceiveCallBack(object sender, MassToolEventArgs e)
        {
            m_dValue = e.mValue;
            if (EventDataReadyEvent != null)
                EventDataReadyEvent(this, new MassToolEventArgs(m_dValue));
        }

        public void LeftDataReceiveCallBack(object sender, MassToolEventArgs e)
        {
            m_dLeft = e.mValue;
            CalResult();

            if (EventDataReadyEvent != null)
                EventDataReadyEvent(this, new MassToolEventArgs(m_dValue));
        }

        public void RightDataReceiveCallBack(object sender, MassToolEventArgs e)
        {
            m_dRight = e.mValue;
            CalResult();

            if (EventDataReadyEvent != null)
                EventDataReadyEvent(this, new MassToolEventArgs(m_dValue));
        }
        #endregion

        #region parameter setting functions
        public OPTYPE mOPType
        {
            get
            {
                return opType;
            }
            set
            {
                opType = value;
            }
        }

        public void SetOPType(char c)
        {
            switch (c)
            {
                case '+':
                    opType = OPTYPE.OP_ADD;
                    break;
                case '-':
                    opType = OPTYPE.OP_SUB;
                    break;
                case '*':
                    opType = OPTYPE.OP_MUL;
                    break;
                case '/':
                    opType = OPTYPE.OP_DIV;
                    break;
                default:
                    opType = OPTYPE.OP_NONE;
                    break;
            }
        }

        public string mPositiveTag
        {
            get
            {
                return m_posTag;
            }
            set
            {
                m_posTag = value;
            }
        }

        public string mNegativeTag
        {
            get
            {
                return m_negTag;
            }
            set
            {
                m_negTag = value;
            }
        }

        public double mValue
        {
            get
            {
                return m_dValue;
            }
        }

        public double mLeftValue
        {
            get
            {
                return m_dLeft;               
            }
            set
            {
                m_dLeft = value;
                CalResult();
            }
        }

        public double mRightValue
        {
            get
            {
                return m_dRight;
            }
            set
            {
                m_dRight = value;
                CalResult();
            }
        }

        public bool IsValue()
        {
            if (mToolEventHandler != null || mLeftEventHandler != null || mRightEventHandler != null)
                return false;
            return true;
        }

        private void CalResult()
        {
            switch (opType)
            {
                case OPTYPE.OP_ADD:
                    m_dValue = m_dLeft + m_dRight;
                    break;
                case OPTYPE.OP_SUB:
                    m_dValue = m_dLeft - m_dRight;
                    break;
                case OPTYPE.OP_MUL:
                    m_dValue = m_dLeft * m_dRight;
                    break;
                case OPTYPE.OP_DIV:
                    m_dValue = m_dLeft / m_dRight;
                    break;
            }
        }
        #endregion

        #region parse the expression
        public static string CreateKey(EventExpr left, EventExpr right, char type)
        {
            return left.mPositiveTag + type + right.mPositiveTag;
        }
        public static EventExpr MergeEvents(EventExpr left, EventExpr right, char type)
        {
            EventExpr result = new EventExpr();
            result.SetOPType(type);
            result.mPositiveTag = CreateKey(left, right, type);

            if (left.IsValue()) result.mLeftValue = left.mValue;
            else left.AddSubscriber(result.GetListener(SIDETYPE.SD_LEFT));

            if (right.IsValue()) result.mRightValue = right.mValue;
            else right.AddSubscriber(result.GetListener(SIDETYPE.SD_RIGHT));

            return result;
        }

        public static EventExpr BuildEventExpr(string expr, Hashtable tbtool, Hashtable tbevents)
        {
            ExprParser parser = new ExprParser(expr, tbtool);
            parser.SetEventTable(tbevents);
            return parser.Parse();
        }
        
        #endregion 
    }
}
