/**
 * Copyright (c) 2009 - Lehigh University. Bethlehem, PA, USA.
 * All rights reserved.
 * This source code is a part of MARCHES Project. 
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL LEHIGH UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT, 
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF LEHIGH
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *	
 * LEHIGH UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, 
 * AND LEHIGH UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 **/

// modsDlg.cpp : implementation file
//

using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Net;
using System.Net.Sockets;
using Masslets;
using System.Reflection;

namespace MassWare
{
    public class Architecture
    {
        public const int MASSWARE_SOCKET_PORT = 9730,
            SYNCHRONIZATION_REQUEST = 255,
            SYNCHRONIZATION_RESPONSE = 254,
            SYNCHRONIZATION_EXIT = 253,
            SYNCHRONIZATION_ASK = 252;

        private Hashtable m_tbMassTools;
        private Hashtable m_tbProMasslets, m_tbReMasslets;

        int m_idxProActs, m_idxReActs;
        List<Actuator> m_listProActs, m_listReActs;
        SyncRequest m_pktRequest;

        UDPClient m_udpC;
        UDPServer m_udpS;
        List<IPAddress> m_listDests;
        event MassletDataReadyEventHandler SocketDataReadyEvent;

        //output results: proactive or reactive architecture change
        public delegate void MassWareArchChangeEventHandler(object sender, MassWareArchEventArgs e);
        public event MassWareArchChangeEventHandler ArchChangeEvent;


        #region constructor and outer functions
        public Architecture()
        {
            m_idxProActs = -1;
            m_idxReActs = -1;
            m_listProActs = new List<Actuator>();
            m_listReActs = new List<Actuator>();
            
            m_tbMassTools = new Hashtable();
            m_tbProMasslets = new Hashtable();
            m_tbReMasslets = new Hashtable();
            
            m_pktRequest = new SyncRequest();

            m_udpC = new UDPClient();
            m_udpS = new UDPServer(MASSWARE_SOCKET_PORT);
            m_listDests = new List<IPAddress>();
        }

        public bool Initialize()
        {
            if (m_tbReMasslets.Count > 0)
            {
                m_udpS.DataReadyEvent += new UDPServer.UDPDataReadyEventHandler(UdpDataReceiveCallBack);
                m_udpS.Start();
            }
            return true;
        }

        public void Dispose()
        {
            IPAddress localAddr = Dns.GetHostEntry(Dns.GetHostName()).AddressList[0];
            foreach (IPAddress ep in m_listDests)
                if (!ep.Equals(localAddr))
                    m_udpC.Send(SYNCHRONIZATION_EXIT, null, ep);
            m_udpC.Close(); 
            m_udpS.Close(); 

            m_listProActs.Clear();
            m_listReActs.Clear();
            m_tbMassTools.Clear();
            m_tbProMasslets.Clear();
            m_tbReMasslets.Clear();

            m_listDests.Clear();
            m_pktRequest.Dispose();
        }

        public bool SyncTo(IPAddress ep)
        {
            //m_udpC.Send(SYNCHRONIZATION_REQUEST, m_pktRequest.Serialize(), ep);
            m_udpC.Send(SYNCHRONIZATION_REQUEST, m_pktRequest.ToByteArray(), ep);
            return true;
        }

        public void StartAppChain()
        {
            // initialize the first pro actuator
            // delete the following code when masstools are ok
            m_idxProActs = 0;
            m_listProActs[m_idxProActs].Start(true);

            if (ArchChangeEvent != null)
                ArchChangeEvent(this, new MassWareArchEventArgs(MassWareArchEventArgs.PROACTIVE_CHANGE, m_listProActs[m_idxProActs].mStructure));
        }

        public void StopAppChain()
        {
            m_listProActs[m_idxProActs].Stop(true);
        }

        public void StartMassTools()
        {
            IDictionaryEnumerator _enumerator = m_tbMassTools.GetEnumerator();
            while (_enumerator.MoveNext())
            {
                MassWare.MassToolReflectObject rfob = (MassWare.MassToolReflectObject)_enumerator.Value;
                rfob.type.InvokeMember(
                    "Start",
                    BindingFlags.InvokeMethod,
                    null,
                    rfob.obj,
                    null);
            }
        }

        public void StopMassTools()
        {
            IDictionaryEnumerator _enumerator = m_tbMassTools.GetEnumerator();
            while (_enumerator.MoveNext())
            {
                MassWare.MassToolReflectObject rfob = (MassWare.MassToolReflectObject)_enumerator.Value;
                rfob.type.InvokeMember(
                    "Stop",
                    BindingFlags.InvokeMethod,
                    null,
                    rfob.obj,
                    null);
            }
        }
        #endregion

        #region set parameters and local variables by xml parser
        public void AddAcutator(EventBC eve, Actuator pact, Actuator ract)
        {
            pact.mUdpClient = m_udpC;
            eve.mIndex = m_listProActs.Count;
            m_listProActs.Add(pact);
            eve.AddSubscriber(new EventHandler(EventReceivingCallBack));
            if (ract != null) m_pktRequest.Add((byte)eve.mIndex, ract);
        }

        public void AddAwareTool(string id, MassWare.MassToolReflectObject rfob)
        {
            m_tbMassTools.Add(id, rfob);
        }

        public void AddProAwarelet(string id, MassWare.MassletReflectObject rfob)
        {
            m_tbProMasslets.Add(id, rfob);
        }

        public void AddReAwarelet(string id, MassWare.MassletReflectObject rfob)
        {
            m_tbReMasslets.Add(id, rfob);
        }

        public void SetAppParameter(string name, string func, object[] paramlist)
        {
            IDictionaryEnumerator _enumerator = m_tbMassTools.GetEnumerator();
            while (_enumerator.MoveNext())
            {
                MassWare.MassToolReflectObject rfob = (MassWare.MassToolReflectObject)_enumerator.Value;
                if (rfob.name == name)
                {
                    rfob.type.InvokeMember(
                        func,
                        BindingFlags.InvokeMethod,
                        null,
                        rfob.obj,
                        paramlist);
                }
            }

            _enumerator = m_tbProMasslets.GetEnumerator();
            while (_enumerator.MoveNext())
            {                
                MassWare.MassletReflectObject rfob = (MassWare.MassletReflectObject)_enumerator.Value;
                if (rfob.name == name)
                {
                    rfob.type.InvokeMember(
                        func,
                        BindingFlags.InvokeMethod,
                        null,
                        rfob.obj,
                        paramlist);
                }
            }

            _enumerator = m_tbReMasslets.GetEnumerator();
            while (_enumerator.MoveNext())
            {
                MassWare.MassletReflectObject rfob = (MassWare.MassletReflectObject)_enumerator.Value;
                if (rfob.name == name)
                {
                    rfob.type.InvokeMember(
                        func,
                        BindingFlags.InvokeMethod,
                        null,
                        rfob.obj,
                        paramlist);
                }
            }
        }

        public Hashtable mMassTools
        {
            get
            {
                return m_tbMassTools;
            }
            set
            {
                m_tbMassTools = value;
            }
        }

        public Hashtable mProMasslets
        {
            get
            {
                return m_tbProMasslets;
            }
            set
            {
                m_tbProMasslets = value;
            }
        }

        public Hashtable mReMasslets
        {
            get
            {
                return m_tbReMasslets;
            }
            set
            {
                m_tbReMasslets = value;
            }
        }
        #endregion


        #region event and socket call back functions
        public void EventReceivingCallBack(object sender, EventArgs e)
        {
            int id = ((EventBC)sender).mIndex;
            if (m_idxProActs == id) return;

            m_listProActs[m_idxProActs].Stop(false);
            m_idxProActs = id;
            m_listProActs[m_idxProActs].Start(false);

            Console.WriteLine("\nproactive actuator change!\n");
            if (ArchChangeEvent != null) 
                ArchChangeEvent(this, new MassWareArchEventArgs(MassWareArchEventArgs.PROACTIVE_CHANGE,m_listProActs[m_idxProActs].mStructure));

        }

        public void UdpDataReceiveCallBack(Object sender, UDPEventArgs e)
        {
            switch (e.m_type)
            {
                case SYNCHRONIZATION_REQUEST:
                    ProcSyncReqestInfo(e.m_ipAddr, e.m_data);
                    break;
                case SYNCHRONIZATION_RESPONSE:
                    ProcSyncResponseInfo(e.m_ipAddr, e.m_data);
                    break;
                case SYNCHRONIZATION_EXIT:
                    ProcSyncExitInfo(e.m_ipAddr, e.m_data);
                    break;
                case SYNCHRONIZATION_ASK:
                    ProcSyncAskInfo(e.m_ipAddr, e.m_data);
                    break;
                default:
                    ProcessPayload(e.m_type, e.m_ipAddr, e.m_data);
                    break;
            }
        }
        #endregion

        #region process received data from socket
        public void ProcSyncReqestInfo(IPAddress addr, byte[] data)
        {
            //SyncRequest pktRequest = SyncRequest.DeSerialize(data);
            SyncRequest pktRequest = SyncRequest.FromByteArray(data);
            SyncReply pktReply = new SyncReply();
            List<SyncRequest.RequestPair> list = pktRequest.mPacket;
            foreach (SyncRequest.RequestPair pair in list)
            {
                Actuator ract = new Actuator();
                ract.mActuator = pair.m_CompntList;
                ract.mSyncType = (Actuator.SYNCTYPE)pair.m_type;
                // set the object instances and parameters
                BuildActuator(ract, m_tbReMasslets);

                // set the sender's address for checking duplicate request packets
                ract.AddDestination(addr, pair.m_idProActuator);

                // check whether this reactive actuator already exists in the list
                byte idx = 0;
                if (m_listReActs.Count == 0) m_listReActs.Add(ract);
                else {
                    for (idx = 0; idx < m_listReActs.Count; idx++)
                    {
                        Actuator.AMDestination dest = m_listReActs[idx].GetFirstDestination();
                        if (dest.endPoint.Equals(addr) && dest.activeHeader == pair.m_idProActuator)
                            break;
                    }
                    if (idx == m_listReActs.Count)
                        m_listReActs.Add(ract);
                    else m_listReActs[idx] = ract;
                }

                pktReply.Add(pair.m_idProActuator, idx);                
            }
            //m_udpC.Send(SYNCHRONIZATION_RESPONSE, pktReply.Serialize(), addr);
            m_udpC.Send(SYNCHRONIZATION_RESPONSE, pktReply.ToByteArray(), addr);
        }

        public void ProcSyncResponseInfo(IPAddress addr, byte[] data)
        {
            //SyncReply pktReply = SyncReply.DeSerialize(data);
            SyncReply pktReply = SyncReply.FromByteArray(data);
            List<SyncReply.ReplyPair> list = pktReply.mPacket;
            foreach (SyncReply.ReplyPair pair in list)
                m_listProActs[pair.m_idProActuator].AddDestination(addr, pair.m_idSyncAcutator);
            m_listDests.Add(addr);
        }

        public void ProcSyncExitInfo(IPAddress addr, byte[] data)
        {
            for (int i = 0; i< m_listReActs.Count; i++) 
            {
                Actuator.AMDestination dest = m_listReActs[i].GetFirstDestination();
                if (dest.endPoint.Equals(addr))
                {
                    m_listReActs.RemoveAt(i);
                    i--;
                }
            }

            // remove destination
            for (int i = 0; i < m_listDests.Count; i++)
            {
                if (m_listDests[i].Equals(addr))
                {
                    m_listDests.RemoveAt(i);
                    i--;
                }
            }

            // if we don't send data to that destination
            foreach (Actuator act in m_listProActs)
                act.RemoveDestination(addr);
        }

        public void ProcSyncAskInfo(IPAddress addr, byte[] data)
        {
            SyncTo(addr);
        }

        public void ProcessPayload(int idx, IPAddress addr, byte[] data)
        {
            if (null == m_listReActs || idx >= m_listReActs.Count)
            {
                m_udpC.Send(SYNCHRONIZATION_ASK, null, addr);
                return;
            }

            if (m_idxReActs != idx)
            {
                if (m_idxReActs >= 0) m_listReActs[m_idxReActs].Stop(false);
                m_idxReActs = idx;
                MassletDataReadyEventHandler[] list = m_listReActs[m_idxReActs].GetSocketEventHandler();

                SocketDataReadyEvent = null;
                foreach (MassletDataReadyEventHandler handler in list)
                    SocketDataReadyEvent += handler;
                m_listReActs[m_idxReActs].Start(false);

                Console.WriteLine("\nreactive actuator change!\n");
                if (ArchChangeEvent != null) 
                    ArchChangeEvent(this, new MassWareArchEventArgs(MassWareArchEventArgs.REACTIVE_CHANGE, m_listReActs[m_idxReActs].mStructure));
            }
            SocketDataReadyEvent(this, new MassletEventArgs(data));
        }

        public void BuildActuator(Actuator act, Hashtable tb)
        {
            for (int i = 0; i < act.mActuator.Count; i++)
            {
                MassWare.MassletReflectObject rfob = act.mActuator[i];
                MassWare.MassletReflectObject orob = (MassWare.MassletReflectObject)tb[rfob.name];
                rfob.type = orob.type;
                rfob.obj = orob.obj;
                IDictionaryEnumerator _enumerator = orob.tbparam.GetEnumerator();
                while (_enumerator.MoveNext())
                    if (!rfob.tbparam.ContainsKey(_enumerator.Key))
                        rfob.tbparam.Add(_enumerator.Key, _enumerator.Value);

                act.mActuator[i] = rfob;
            }
        }
        #endregion
    }

    public class MassWareArchEventArgs : EventArgs
    {
        public const byte PROACTIVE_CHANGE = 0, REACTIVE_CHANGE = 1;
        private byte type;
        private string message;
        public MassWareArchEventArgs(byte t, string msg)
        {
            type = t;
            message = msg;
        }
        public byte mType
        {
            get
            {
                return type;
            }
        }
        public string mMessage
        {
            get
            {
                return message;
            }
        }
    }
}
